% KM-BL code for recreating Figure 5 in Morrison et al. Indoor Air. 29,
% 956–967. (2019) 


%Layers
% 1: L_far =OH boundary layer further from the surface, 
% L_far+1:L_far+L_close= OH boundary layer near the surface, 
% L_far+L_close+1= OH at a distance of 1 mean free path from the surface
% L_far+L_close+2= OH on the surface
% L_far+ L_close +3: 2*L_far+L_close+2= O3 boundary layer further from the surface, 
% 2*L_far+L_close+3: 2*L_far+2*L_close+2=O3 boundary layer near the surface, 
% 2*L_far+2*L_close+3= O3 at a distance of 1 mean free path from the surface
% 2*L_far+2*L_close+4= O3 on the surface
% 2*L_far+ 2*L_close +5: 3*L_far+2*L_close+4= terpene boundary layer further from the surface, 
% 3*L_far+2*L_close+5: 3*L_far+3*L_close+4=terpene boundary layer near the surface, 
% 3*L_far+3*L_close+5= terpene at a distance of 1 mean free path from the surface
% 3*L_far+3*L_close+6= terpene on the surface

%Gas-phase reactions
% R1: O3 + Limonene -> Y OH
% R2: OH + TLimonene -> Other products
% R3: OH -> Other products

clear all
warning off all
global mfp_OH L_far L_close a0_OH a0_O3 Dg_OH Gas_delta_far Gas_delta_close A
global V_close V_far W Dg_O3 kgr Yield OH_uptake O3_uptake mfp_O3 
global a0_terpene terpene_uptake Dg_terpene mfp_terpene

Gamma_O3=[1e-6,1e-4,1e-2,1]; %O3 uptake coefficients
for NN=1:4
    Run_number=NN

%%%%%%%--------------General Parameters-----------------%%%%%%%%%%

T=273+25; %Temperature (K)
pi=3.14159;
R= 8.31e7; %Gas constant (g cm2 s-2 K-1 mol-1)
NA=6.02e23; %Avogadro's number (mol-1)

%%%%%%%--------------Parameters for different species-----------------%%%%%%%%%%
% ------Parameters for OH------

Mx(1)=18; % Molar mass of OH (g mol-1)
W(1)=sqrt(8*R*T/pi/Mx(1)); % Mean thermal velocity of OH (cm s-1)
Dg_OH=0.24; % OH gas-phase diffusion coefficient (cm2 s-1)
mfp_OH= ((3*Dg_OH)/W(1)); % OH mean free path (cm)
a0_OH=1; % OH mass accommodation coefficient on an adsorbate free surface
OH_uptake= 1; % OH uptake coefficient

% ------Parameters for O3------

Mx(2)=48; % Molar mass of O3 (g mol-1)
W(2)=sqrt(8*R*T/pi/Mx(2)); % Mean thermal velocity of O3 (cm s-1)
Dg_O3=0.16; % O3 gas-phase diffusion coefficient (cm2 s-1)
mfp_O3= ((3*Dg_O3)/W(2)); % O3 mean free path (cm)
a0_O3=1; % O3 mass accommodation coefficient on an adsorbate free surface
O3_uptake=Gamma_O3(NN); % O3 uptake coefficient

% ------Parameters for terpene------

Mx(3)=136.24; % Molar mass of limonene (g mol-1)
W(3)=sqrt(8*R*T/pi/Mx(3)); % Mean thermal velocity of limonene (cm s-1)
Dg_terpene=0.06; % Limonene gas-phase diffusion coefficient (cm2 s-1)
mfp_terpene= ((3*Dg_terpene)/W(3)); % Limonene mean free path (cm)
a0_terpene=1; % Limonene mass accommodation coefficient on an adsorbate free surface
terpene_uptake=0; % Limonene uptake coefficient

% ---------Geometric factors------------

L_far=1000; %Number of layers in the boundary layer further away from the surface
L_close=500; %Number of layers in the close boundary layer
GasR_far=10; %Thickness of the boundary layer further away from the surface %%NOTE: V.IMPORTANT needs to be varied until there are no conc. gradients at furthest distances + also no sudden conc. changes, must be smooth otherwise gives the wrong result
GasR_close=0.005; % Thickness of the boundary layer close to the surface (cm)
Gas_delta_far=GasR_far/L_far; %Thickness of 1 layer in the boundary layer further away from the surface (cm)
Gas_delta_close=GasR_close/L_close; %Thickness of 1 layer in the close boundary layer (cm)

A=1; %Unit surface area of the boundary layer (cm2)

V_close= A*Gas_delta_close; %Volume of 1 layer in the close boundary layer (cm3)
V_far= A*Gas_delta_far; %Volume of 1 layer in the boundary layer further away from the surface (cm3)

%%%%%%%%%Diffusion parameters%%%%%%%%%

%Distance from the surface to the top of a layer (cm):
Distance_to_between_layers(1,1)=mfp_OH+GasR_close+GasR_far-1*Gas_delta_far;
for i=2:L_far
Distance_to_between_layers(i,1)=mfp_OH+GasR_close+GasR_far-1*Gas_delta_far-(i-1)*Gas_delta_far;
end
Distance_to_between_layers(L_far+1,1)=mfp_OH+GasR_close-1*Gas_delta_close;
for i=L_far+2:L_far+L_close
Distance_to_between_layers(i,1)=mfp_OH+GasR_close-1*Gas_delta_close-(i-L_far-1)*Gas_delta_close;
end

%Note: if Turbulent_diffusion=1 then turbulent diffusion is occuring, if
%Turbulent_diffusion=0 then only Fickian diffusion is occuring.

Turbulent_diffusion=1; 

if Turbulent_diffusion==0
    for i=1:L_far+L_close
Dg_OH(i)=0.24; % OH gas-phase diffusion coefficient (cm2 s-1)
Dg_O3(i)=0.16; % O3 gas-phase diffusion coefficient (cm2 s-1)
Dg_terpene(i)=0.06; % Limonene gas-phase diffusion coefficient (cm2 s-1)
    end
elseif Turbulent_diffusion==1
    Ke=0.1; %Turbulence intensity (s-1)
    for i=1:L_far+L_close
        Ep(i)=Ke.*(Distance_to_between_layers(i,1))^2; %Eddy diffusion coefficient (cm2 s-1)
Dg_OH(i)=0.24+Ep(i); % OH gas-phase diffusion coefficient (cm2 s-1)
Dg_O3(i)=0.16+Ep(i); % O3 gas-phase diffusion coefficient (cm2 s-1)
Dg_terpene(i)=0.06+Ep(i); % Limonene gas-phase diffusion coefficient (cm2 s-1)
    end
end


%-----Rate coefficients---------
kgr(1)=2e-16;%Reaction rate coefficient of O3 + Limonene (cm3 s-1)
kgr(2)= 1.7e-10; %Reaction rate coefficient of OH + Limonene (cm3 s-1)
kgr(3)= 50;% First-order OH background loss (s-1)
Yield=0.67; % yield of OH from the reaction of O3 + Limonene

%%%%%%%-------------Concentrations---------------%%%%%%

y0=zeros(3*L_close+3*L_far+6,1);
for i=L_far+L_close+3:2*L_far+2*L_close+3
    y0(i)=10*2.46e10; %Ozone initial gas-phase concentrations (cm-3)
end

for i=2*L_far+ 2*L_close +5:3*L_far+3*L_close+5
    y0(i)=10*2.46e10; %Limonene initial gas-phase concentrations (cm-3)
end

y0(1)=10*2.46e10*kgr(1)*10*2.46e10*Yield/(kgr(2)*10*2.46e10+kgr(3)); %OH initial gas-phase concentrations (cm-3) (based on steady state)


%%%%%%%--------Solve differential equations------------%%%%%%%
n=9999; %outputted time steps
t=logspace(-3,log10(3600*12),n);t=t'; %time being investigated (s)
AbsTol=1;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23tb(@KM_BL_F,t,y0,options); %solver

%%%%%%-------------Outputs-----------%%%%%
%Distance from the surface to the middle of a layer (cm):
Distance_output(1,1)=mfp_OH+GasR_close+GasR_far-0.5*Gas_delta_far;
for i=2:L_far
Distance_output(i,1)=mfp_OH+GasR_close+GasR_far-0.5*Gas_delta_far-(i-1)*Gas_delta_far;
end
Distance_output(L_far+1,1)=mfp_OH+GasR_close-0.5*Gas_delta_close;
for i=L_far+2:L_far+L_close
Distance_output(i,1)=mfp_OH+GasR_close-0.5*Gas_delta_close-(i-L_far-1)*Gas_delta_close;
end
Distance_output(L_far+L_close+1,1)=mfp_OH;

for i=1:L_close+L_far+1
OH_conc_output(i,NN)=Y(n,i); % OH concentration in the boundary layer (cm-3)
O3_conc_output(i,NN)=Y(n,L_far+L_close+2+i); % O3 concentration in the boundary layer (cm-3)
limonene_conc_output(i,NN)=Y(n,2*L_far+2*L_close+4+i); % Limonene concentration in the boundary layer (cm-3)
end
end

%%%%-----Figures to be outputted----%%%%%

figure(1)
loglog(Distance_output(:,1),OH_conc_output(:,1),'k');
hold on
loglog(Distance_output(:,1),OH_conc_output(:,2),'r');
hold on
loglog(Distance_output(:,1),OH_conc_output(:,3),'b');
hold on
loglog(Distance_output(:,1),OH_conc_output(:,4),'m');
xlabel('Height above the surface (cm)')
ylabel('OH conc. (cm-3)')
legend('Gamma O3 = 1e-6','Gamma O3 = 1e-4','Gamma O3 = 1e-2','Gamma O3 = 1')

figure(2)
loglog(Distance_output(:,1),O3_conc_output(:,1),'k');
hold on
loglog(Distance_output(:,1),O3_conc_output(:,2),'r');
hold on
loglog(Distance_output(:,1),O3_conc_output(:,3),'b');
hold on
loglog(Distance_output(:,1),O3_conc_output(:,4),'m');
xlabel('Height above the surface (cm)')
ylabel('O3 conc. (cm-3)')
legend('Gamma O3 = 1e-6','Gamma O3 = 1e-4','Gamma O3 = 1e-2','Gamma O3 = 1')



